/*!
 * ImmobilienScout24 Core JS, v2.0.2
 * http://www.immobilienscout24.de/
 */
var IS24 = IS24 || {};
IS24.core = IS24.core || {};

(function () {
"use strict";

/* --- Helper Functions for Core Framework Scripts --- */

var helpers = {
	documentReady: function(callback) {
		if (document.addEventListener) {
			document.addEventListener('DOMContentLoaded', callback);
		} else if (document.attachEvent) {
			document.attachEvent('onreadystatechange', function() {
				if (document.readyState === 'complete') {
					callback();
				}
			});
		}
	},
	
	addEventListener: function (targetElement, eventType, handler, useCapture) {
		if (targetElement.addEventListener) {
			targetElement.addEventListener(eventType, handler, useCapture);
			return true;
		} else if (targetElement.attachEvent) {
			return targetElement.attachEvent("on" + eventType, handler);
		}
	},
	
	/* 
	 * @returns obj { top: <int TopOffset>, left: <int LeftOffset>}
	 */
	getPageScroll: function () {
		var scroll = { top: 0, left: 0 };
		
		if ("scrollTop" in document.documentElement && "scrollLeft" in document.documentElement) {
			// get sum of document and body offset so it doesn't matter which of the two the browser scrolls - either one should be zero
			scroll.top  = document.documentElement.scrollTop  + document.body.scrollTop;
			scroll.left = document.documentElement.scrollLeft + document.body.scrollLeft;
		} else if ("pageYOffset" in document.body && "pageXOffset" in document.body) {
			scroll.top  = document.body.pageYOffset;
			scroll.left = document.body.pageXOffset;
		}
		
		return scroll;
	},
	
	/* 
	 * Determines coordinates of an element within the document.
	 * @returns obj { x: <int horizontalCoordinate>, y: <int verticalCoordinate>}
	 */
	getCoordinates: function (element) {
		var boundingRect = (element.getBoundingClientRect() ? element.getBoundingClientRect() : { top: 0, left: 0 }),
			coordinates = {};
		
		coordinates.x = parseInt(boundingRect.left + this.getPageScroll().left, 10);
		coordinates.y = parseInt(boundingRect.top  + this.getPageScroll().top, 10);
		
		return coordinates;
	},
	
	hasClass: function (element, wantedClass) {
		var classString,
			classMatch;
		
		if (!element || !wantedClass) { return; }
		
		classString = (" " + element.className + " ").replace(/\s+/g, " "); // makes sure all class names are padded by single white spaces
		classMatch = new RegExp("\\s" + wantedClass + "\\s", "g");
		
		return !!classString.match(classMatch);
	},
	
	addClass: function (element, addedClass) {
		var elementHasClass = this.hasClass(element, addedClass);
		
		if (!elementHasClass) {
			element.className += " " + addedClass;
		}
	},
	
	removeClass: function (element, removedClass) {
		var elementHasClass = this.hasClass(element, removedClass);
		
		if (elementHasClass) {
			element.className = (" " + element.className + " ").replace(" " + removedClass + " ", " ").replace(/^\s+|\s+$/g, ''); // remove class, trim
		}
	},
	
	toggleClass: function (element, toggledClass) {
		var elementHasClass = this.hasClass(element, toggledClass);
		
		if (elementHasClass) {
			this.removeClass(element, toggledClass);
		} else {
			this.addClass(element, toggledClass);
		}
	},
	
	nearestAncestor: function (element, ancestorSelector) {
		var currentNode = element,
			matchedAncestor,
			isClassSelector = (ancestorSelector.indexOf(".") === 0);
		
		ancestorSelector = (isClassSelector) ? ancestorSelector.substring(1) : ancestorSelector;
		
		if (isClassSelector) {
			while (currentNode.parentNode) {
				currentNode = currentNode.parentNode;
				if (this.hasClass(currentNode, ancestorSelector)) {
					matchedAncestor = currentNode;
					break;
				}
			}
		} else {
			while (currentNode.parentNode) {
				currentNode = currentNode.parentNode;
				if (currentNode.nodeName.toLowerCase() === ancestorSelector.toLowerCase()) {
					matchedAncestor = currentNode;
					break;
				}
			}
		}
		
		return matchedAncestor;
	},
	
	/*
	 * @returns obj { hostAndPath: string, search: string, hash: string }
	 * query and fragment are called "search" and "hash" for compatibility with window.location
	 */
	resolveURLString: function(URLString) {
		var hasSearch = (URLString.indexOf("?") !== -1),
			hasHash = (URLString.indexOf("#") !== -1);
		
		return {
			hostAndPath: (hasSearch) ? URLString.split("?")[0] : URLString.split("#")[0],
			search: (hasSearch) ? URLString.substring(URLString.indexOf("?")).split("#")[0] : "",
			hash: (hasHash) ? URLString.substring(URLString.indexOf("#")) : ""
		};
	},
	
	/* 
	 * Attaches an arbitrary number of URL parameters to the href attribute of an element, overwrites previous occurrences
	 * @argument DOMElement element
	 * @argument obj addedParameters { parameterName: parameterValue }
	 */
	attachLinkParameters: function(element, addedParameters) {
		var sanitizeRegExp = /[^\w\-\s.,+%@&="\[\]]/g,
			sanitizedAddedParameters = {},
			sanitizedName,
			sanitizedValue,
			resolvedHref,
			currentParameters,
			currentParameterName,
			parameterString,
			i, key;
		
		if (!(element && element.href) || !addedParameters || typeof addedParameters !== "object") { return; }
		
		// sanitize input
		for (key in addedParameters) {
			if (addedParameters.hasOwnProperty(key)) {
				sanitizedName = key.replace(sanitizeRegExp, "");
				sanitizedValue = addedParameters[key].replace(sanitizeRegExp, "");
				sanitizedAddedParameters[sanitizedName] = sanitizedValue;
			}
		}
		
		// resolve href, separate parameters
		resolvedHref = this.resolveURLString(element.href);
		currentParameters = (resolvedHref.search) ? resolvedHref.search.substring(1).split("&") : [];
		
		// remove duplicates from current parameters
		for (i=0; i<currentParameters.length; i++) {
			currentParameterName = currentParameters[i].split("=")[0];
			if (sanitizedAddedParameters[currentParameterName]) {
				currentParameters.splice(i, 1);
			}
		}
		
		// add new parameters, stringify
		for (key in sanitizedAddedParameters) {
			if (sanitizedAddedParameters.hasOwnProperty(key)) {
				currentParameters.push(key + "=" + sanitizedAddedParameters[key]);
			}
		}
		parameterString = "?" + currentParameters.join("&");
		
		// insert
		element.href = resolvedHref.hostAndPath + parameterString + resolvedHref.hash;
	}
};
/* --- Public Helpers --- */
// (mainly for mocking in unit tests)

IS24.core.helpers = {
	loadScript: function (url) {
		var scriptElement = document.createElement('script');
		scriptElement.type = "text/javascript";
		scriptElement.src = url;
		document.getElementsByTagName('head')[0].appendChild(scriptElement);
	},
	
	getLocation: function () {
		return window.location;
	}
};
/* --- Fixed Elements --- */
/*
 *  Will make an element stick to the top of the window when the browser scrolls past it.
 *  Initiated through "IS24.core.fixElementToViewportTop.init()".
 *  Required Markup:
 *  <div class="fix-to-top-wrapper">
 *      <div class="fix-to-top"></div>
 *  </div>
 */

IS24.core.fixElementToViewportTop = (function () {
	var fixedElementWrapper,
		fixedElement,
		isFixedClass,
		fixedPoint,
		currentOffset,
		isFixed = false,
		throttle,
		throttleTimeout = 1000;
	
	function updateCurrentOffset() {
		currentOffset = helpers.getPageScroll().top;
	}
	function updateFixedPoint() {
		fixedPoint = helpers.getCoordinates(fixedElementWrapper).y;
	}
	
	function toggleFixed() {
		if (currentOffset > fixedPoint && !isFixed) {
			helpers.addClass(fixedElement, isFixedClass);
			isFixed = true;
		} else if (currentOffset < fixedPoint && isFixed) {
			helpers.removeClass(fixedElement, isFixedClass);
			isFixed = false;
		}
	}
	
	// selective scroll handler to reduce computing time if scroll events are fired in rapid succession
	function throttledHandler() {
		if (!throttle) {
			throttle = window.setTimeout(function () {
				throttle = null;
			}, throttleTimeout);
			
			// put stuff here that should only run every so often
			updateFixedPoint();
		}
		// put stuff here that needs to run every time for responsiveness
		updateCurrentOffset();
		toggleFixed();
	}
	
	function init() {
		fixedElementWrapper = document.querySelector(".fix-to-top-wrapper");
		fixedElement = document.querySelector(".fix-to-top");
		isFixedClass = "fix-to-top--fixed";
		
		if (!fixedElementWrapper || !fixedElement) { return; }
		
		helpers.addEventListener(window, "resize", throttledHandler);
		helpers.addEventListener(window, "scroll", throttledHandler);
		
		// initial execution
		updateFixedPoint();
		updateCurrentOffset();
		toggleFixed();
	}
	
	return { init: init };
}());

/* --- SSO Login Status --- */

IS24.core.sso = (function () {
	function getFirstAndSecondLevelDomain() {
		var host = IS24.core.helpers.getLocation().hostname,
			hostWithoutFirstLevel,
			secondLevelDotIndex;
		
		// trim leading and trailing dots
		while (host.slice(0,1) === ".") { host = host.slice(1); }
		while (host.slice( -1) === ".") { host = host.slice(0, host.length - 1); }
		
		if (host.indexOf(".") !== -1) {
			hostWithoutFirstLevel = host.substring(0, host.lastIndexOf(".") - 1);
			secondLevelDotIndex = hostWithoutFirstLevel.lastIndexOf(".");
			if (secondLevelDotIndex !== -1) {
				host = host.substring(secondLevelDotIndex + 1);
			}
		}
		
		return host;
	}
	
	function getSSOApiDomain() {
		return "sso." + getFirstAndSecondLevelDomain();
	}
	
	function appendSSOLinkParameters(options) {
		var loginLink = document.getElementById("link_loginLinkInternal"),
			logoutLink = document.getElementById("link_logoutLinkInternal"),
			registerLink = document.getElementById("link_registerLinkInternal"),
			returnLocation = encodeURIComponent(options.ssoReturnUrl),
			ssoAppName = encodeURIComponent(options.ssoAppName);
		
		if (loginLink) {
			helpers.attachLinkParameters(loginLink, {
				"returnUrl": returnLocation,
				"appName": ssoAppName
			});
		}
		
		if (logoutLink) {
			helpers.attachLinkParameters(logoutLink, {
				"openid.ssoff.rp_post_logout_url": returnLocation
			});
		}
		
		if (registerLink) {
			helpers.attachLinkParameters(registerLink, {
				"sso_return": returnLocation,
				"appName": ssoAppName
			});
		}
	}
	
	function getJSONPUrl(callbackName) {
		var apiHost = getSSOApiDomain(),
			timestamp = Math.round((new Date()).getTime() / 1000); // cache killer
		
		return "https://" + apiHost + "/sso/getLoginInfo?callback=" + callbackName + "&ts=" + timestamp;
	}
	
	function updateLoginStatus() {
		var options = {
			ssoReturnUrl: (IS24.ssoReturnUrl || IS24.core.helpers.getLocation().href),
			ssoAppName: (IS24.ssoAppName || "is24main")
		};
		
		appendSSOLinkParameters(options);
		IS24.core.helpers.loadScript(getJSONPUrl("IS24.core.sso.jsonpCallback"));
	}
	
	function jsonpCallback(data) {
		var loginWrapper = document.querySelector(".sso-login"),
			userNameContainer = document.querySelector(".sso-login__user-name"),
			loggedInClass = "sso-login--logged-in";
		
		if (!(data && data.username) || !loginWrapper || !userNameContainer) { return; }
		
		userNameContainer.innerHTML = data.username;
		helpers.addClass(loginWrapper, loggedInClass);
	}
	
	return {
		updateLoginStatus: updateLoginStatus,
		jsonpCallback: jsonpCallback
	};
}());

/* --- Top Navigation --- */

IS24.core.topNavigation = {
	slideIn: (function () {
		var leftTrigger,
			rightTrigger,
			slideInContainer,
			slideInContainerIsOpenedClassFragment;
		
		function clickHandler(event, direction) {
			event.preventDefault();
			helpers.toggleClass(slideInContainer, slideInContainerIsOpenedClassFragment + direction);
		}
		
		function documentHandler(event) {
			var target = event.target || event.srcElement;
			if (helpers.nearestAncestor(target, ".top-navigation__slide-in--left") ||
				helpers.nearestAncestor(target, ".top-navigation__slide-in--right")) {
				event.stopPropagation();
			} else if (helpers.hasClass(slideInContainer, slideInContainerIsOpenedClassFragment + "left")) {
				event.preventDefault();
				helpers.removeClass(slideInContainer, slideInContainerIsOpenedClassFragment + "left");
			} else if (helpers.hasClass(slideInContainer, slideInContainerIsOpenedClassFragment + "right")) {
				event.preventDefault();
				helpers.removeClass(slideInContainer, slideInContainerIsOpenedClassFragment + "right");
			}
		}
		
		function init() {
			leftTrigger = document.querySelector(".top-navigation__slide-in-trigger--left");
			rightTrigger = document.querySelector(".top-navigation__slide-in-trigger--right");
			slideInContainer = document.querySelector(".top-navigation-slide-in-container");
			slideInContainerIsOpenedClassFragment = "top-navigation-slide-in-container--open-";
			
			if (!slideInContainer) { return; }
			
			if (leftTrigger) {
				helpers.addEventListener(leftTrigger, "click", function (event) {
					clickHandler(event, "left");
				});
			}
			if (rightTrigger) {
				helpers.addEventListener(rightTrigger, "click", function (event) {
					clickHandler(event, "right");
				});
			}
			if (leftTrigger || rightTrigger) {
				helpers.addEventListener(document, "touchend", function (event) {
					documentHandler(event);
				});
			}
		}
		
		return { init: init };
	}()),
	
	touchHover: (function () {
		var hoverTabs,
			hoverTabLabels,
			pseudoHoverClass,
			lastTouchedItem,
			deskBreakPoint = 1014;
		
		// prevent first click and simulate hover instead
		function tabHandler(event) {
			var ancestorLIElement,
				i;
			
			if (!window.matchMedia("(min-width: " + deskBreakPoint + "px)").matches) { return; } // skip touch handling when only mobile-nav is displayed
			
			if (lastTouchedItem !== event.target) {
				lastTouchedItem = event.target;
				event.preventDefault();
				event.stopPropagation();
				for (i=0; i<hoverTabs.length; i++) {
					helpers.removeClass(hoverTabs[i], pseudoHoverClass);
				}
				ancestorLIElement = helpers.nearestAncestor(event.target, "li");
				helpers.addClass(ancestorLIElement, pseudoHoverClass);
			}
		}
		
		// remove hover effect on touch events outside of layer
		function documentHandler() {
			var i;
			
			if (!lastTouchedItem) { return; }
			
			lastTouchedItem = undefined;
			for (i=0; i<hoverTabs.length; i++) {
				helpers.removeClass(hoverTabs[i], pseudoHoverClass);
			}
		}
		
		function attachEventListeners() {
			var i;
			for (i=0; i<hoverTabLabels.length; i++) {
				helpers.addEventListener(hoverTabLabels[i], "touchstart", tabHandler);
			}
			helpers.addEventListener(document, "touchstart", documentHandler);
		}
		
		function init() {
			hoverTabs = document.querySelectorAll(".top-navigation__hover-tabs > li");
			hoverTabLabels = document.querySelectorAll(".top-navigation__hover-tabs__label");
			pseudoHoverClass = "hover";
			
			if (hoverTabLabels.length === 0 || hoverTabs.length === 0) { return; }
			
			attachEventListeners();
		}
		
		return { init: init };
	}()),
	
	init: function () {
		this.slideIn.init();
		this.touchHover.init();
	}
};

}());